/*
 * Decompiled with CFR 0.152.
 */
package io.dropwizard.servlets.assets;

import io.dropwizard.servlets.assets.ByteRange;
import io.dropwizard.servlets.assets.ResourceURL;
import io.dropwizard.util.Resources;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.CRC32;
import javax.annotation.Nullable;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

public class AssetServlet
extends HttpServlet {
    private static final long serialVersionUID = 6393345594784987908L;
    private static final String IF_MODIFIED_SINCE = "If-Modified-Since";
    private static final String IF_NONE_MATCH = "If-None-Match";
    private static final String IF_RANGE = "If-Range";
    private static final String RANGE = "Range";
    private static final String ACCEPT_RANGES = "Accept-Ranges";
    private static final String CONTENT_RANGE = "Content-Range";
    private static final String ETAG = "ETag";
    private static final String LAST_MODIFIED = "Last-Modified";
    private static final String DEFAULT_MEDIA_TYPE = "text/html";
    private final String resourcePath;
    private final String uriPath;
    @Nullable
    private final String indexFile;
    @Nullable
    private final Charset defaultCharset;

    public AssetServlet(String resourcePath, String uriPath, @Nullable String indexFile, @Nullable Charset defaultCharset) {
        String trimmedPath = AssetServlet.trimSlashes(resourcePath);
        this.resourcePath = trimmedPath.isEmpty() ? trimmedPath : trimmedPath + '/';
        String trimmedUri = AssetServlet.trimTrailingSlashes(uriPath);
        this.uriPath = trimmedUri.isEmpty() ? "/" : trimmedUri;
        this.indexFile = indexFile;
        this.defaultCharset = defaultCharset;
    }

    private static String trimSlashes(String s) {
        Matcher matcher = Pattern.compile("^/*(.*?)/*$").matcher(s);
        if (matcher.find()) {
            return matcher.group(1);
        }
        return s;
    }

    private static String trimTrailingSlashes(String s) {
        Matcher matcher = Pattern.compile("(.*?)/*$").matcher(s);
        if (matcher.find()) {
            return matcher.group(1);
        }
        return s;
    }

    public URL getResourceURL() {
        return Resources.getResource((String)this.resourcePath);
    }

    public String getUriPath() {
        return this.uriPath;
    }

    @Nullable
    public String getIndexFile() {
        return this.indexFile;
    }

    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        try {
            String ifRange;
            CachedAsset cachedAsset;
            StringBuilder builder = new StringBuilder(req.getServletPath());
            if (req.getPathInfo() != null) {
                builder.append(req.getPathInfo());
            }
            if ((cachedAsset = this.loadAsset(builder.toString())) == null) {
                resp.sendError(404);
                return;
            }
            if (this.isCachedClientSide(req, cachedAsset)) {
                resp.sendError(304);
                return;
            }
            String rangeHeader = req.getHeader(RANGE);
            int resourceLength = cachedAsset.getResource().length;
            List<Object> ranges = Collections.emptyList();
            boolean usingRanges = false;
            if (rangeHeader != null && ((ifRange = req.getHeader(IF_RANGE)) == null || cachedAsset.getETag().equals(ifRange))) {
                try {
                    ranges = this.parseRangeHeader(rangeHeader, resourceLength);
                }
                catch (NumberFormatException e) {
                    resp.sendError(416);
                    return;
                }
                if (ranges.isEmpty()) {
                    resp.sendError(416);
                    return;
                }
                resp.setStatus(206);
                usingRanges = true;
                String byteRanges = ranges.stream().map(ByteRange::toString).collect(Collectors.joining(","));
                resp.addHeader(CONTENT_RANGE, "bytes " + byteRanges + "/" + resourceLength);
            }
            resp.setDateHeader(LAST_MODIFIED, cachedAsset.getLastModifiedTime());
            resp.setHeader(ETAG, cachedAsset.getETag());
            String mediaType = Optional.ofNullable(req.getServletContext().getMimeType(req.getRequestURI())).orElse(DEFAULT_MEDIA_TYPE);
            if (mediaType.startsWith("video") || mediaType.startsWith("audio") || usingRanges) {
                resp.addHeader(ACCEPT_RANGES, "bytes");
            }
            resp.setContentType(mediaType);
            if (this.defaultCharset != null) {
                resp.setCharacterEncoding(this.defaultCharset.toString());
            }
            try (ServletOutputStream output = resp.getOutputStream();){
                if (usingRanges) {
                    for (ByteRange byteRange : ranges) {
                        output.write(cachedAsset.getResource(), byteRange.getStart(), byteRange.getEnd() - byteRange.getStart() + 1);
                    }
                } else {
                    output.write(cachedAsset.getResource());
                }
            }
        }
        catch (RuntimeException | URISyntaxException ignored) {
            resp.sendError(404);
        }
    }

    @Nullable
    private CachedAsset loadAsset(String key) throws URISyntaxException, IOException {
        long lastModified;
        if (!key.startsWith(this.uriPath)) {
            throw new IllegalArgumentException("Cache key must start with " + this.uriPath);
        }
        String requestedResourcePath = AssetServlet.trimSlashes(key.substring(this.uriPath.length()));
        String absoluteRequestedResourcePath = AssetServlet.trimSlashes(this.resourcePath + requestedResourcePath);
        URL requestedResourceURL = this.getResourceUrl(absoluteRequestedResourcePath);
        if (ResourceURL.isDirectory(requestedResourceURL)) {
            if (this.indexFile != null) {
                requestedResourceURL = this.getResourceUrl(absoluteRequestedResourcePath + '/' + this.indexFile);
            } else {
                return null;
            }
        }
        if ((lastModified = ResourceURL.getLastModified(requestedResourceURL)) < 1L) {
            lastModified = System.currentTimeMillis();
        }
        lastModified = lastModified / 1000L * 1000L;
        return new CachedAsset(this.readResource(requestedResourceURL), lastModified);
    }

    protected URL getResourceUrl(String absoluteRequestedResourcePath) {
        return Resources.getResource((String)absoluteRequestedResourcePath);
    }

    protected byte[] readResource(URL requestedResourceURL) throws IOException {
        return Resources.toByteArray((URL)requestedResourceURL);
    }

    private boolean isCachedClientSide(HttpServletRequest req, CachedAsset cachedAsset) {
        return cachedAsset.getETag().equals(req.getHeader(IF_NONE_MATCH)) || req.getDateHeader(IF_MODIFIED_SINCE) >= cachedAsset.getLastModifiedTime();
    }

    private List<ByteRange> parseRangeHeader(String rangeHeader, int resourceLength) {
        String[] parts;
        List<ByteRange> byteRanges = rangeHeader.contains("=") ? ((parts = rangeHeader.split("=", -1)).length > 1 ? Arrays.stream(parts[1].split(",", -1)).map(String::trim).map(s -> ByteRange.parse(s, resourceLength)).collect(Collectors.toList()) : Collections.emptyList()) : Collections.emptyList();
        return byteRanges;
    }

    private static class CachedAsset {
        private final byte[] resource;
        private final String eTag;
        private final long lastModifiedTime;

        private CachedAsset(byte[] resource, long lastModifiedTime) {
            this.resource = resource;
            this.eTag = '\"' + CachedAsset.hash(resource) + '\"';
            this.lastModifiedTime = lastModifiedTime;
        }

        private static String hash(byte[] resource) {
            CRC32 crc32 = new CRC32();
            crc32.update(resource);
            return Long.toHexString(crc32.getValue());
        }

        public byte[] getResource() {
            return this.resource;
        }

        public String getETag() {
            return this.eTag;
        }

        public long getLastModifiedTime() {
            return this.lastModifiedTime;
        }
    }
}

